<?php
ob_start();
session_start();
require '../conexion.php';

// Verificar si el usuario está logueado y es administrador
if (!isset($_SESSION['username']) || !isset($_SESSION['rol']) || ($_SESSION['rol'] !== 'admin' && $_SESSION['rol'] !== 'empleado')) {
    header("Location: ../login.php");
    exit;
}

// Procesar la creación de producto
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['crear_producto'])) {
    $nombre = $_POST['nombre'];
    $descripcion = $_POST['descripcion'];
    $precio = $_POST['precio'];
    $stock = $_POST['stock'];
    $destacado = isset($_POST['destacado']) ? 1 : 0;
    
    // Manejar la imagen
    $imagen = '';
    if (isset($_FILES['imagen']) && $_FILES['imagen']['error'] == 0) {
        $imagen_nombre = basename($_FILES['imagen']['name']);
        $imagen_temp = $_FILES['imagen']['tmp_name'];
        $carpeta_destino = "../img/productos/";
        
        // Crear la carpeta si no existe
        if (!file_exists($carpeta_destino)) {
            mkdir($carpeta_destino, 0777, true);
        }
        
        if (move_uploaded_file($imagen_temp, $carpeta_destino . $imagen_nombre)) {
            $imagen = $imagen_nombre;
        } else {
            $error = "Error al subir la imagen.";
        }
    }
    
    // Insertar el producto en la base de datos
    if (!isset($error)) {
        $query = "INSERT INTO productos (nombre, descripcion, precio, imagen, stock, destacado) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("ssdsii", $nombre, $descripcion, $precio, $imagen, $stock, $destacado);
        
        if ($stmt->execute()) {
            $success_message = "Producto creado correctamente.";
        } else {
            $error = "Error al crear el producto: " . $conn->error;
        }
    }
}

// Procesar la eliminación de producto
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['eliminar_producto'])) {
    $producto_id = $_POST['producto_id'];
    
    // Obtener la información de la imagen
    $query_imagen = "SELECT imagen FROM productos WHERE id = ?";
    $stmt_imagen = $conn->prepare($query_imagen);
    $stmt_imagen->bind_param("i", $producto_id);
    $stmt_imagen->execute();
    $result_imagen = $stmt_imagen->get_result();
    $producto = $result_imagen->fetch_assoc();
    
    // Eliminar el producto
    $query = "DELETE FROM productos WHERE id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $producto_id);
    
    if ($stmt->execute()) {
        // Eliminar la imagen si existe
        if (!empty($producto['imagen'])) {
            $ruta_imagen = "../img/productos/" . $producto['imagen'];
            if (file_exists($ruta_imagen)) {
                unlink($ruta_imagen);
            }
        }
        $success_message = "Producto eliminado correctamente.";
    } else {
        $error = "Error al eliminar el producto: " . $conn->error;
    }
}

// Obtener la lista de productos
$query_productos = "SELECT * FROM productos ORDER BY id DESC";
$result_productos = mysqli_query($conn, $query_productos);

// Obtener el nombre del rol para mostrar en la interfaz
$rol_usuario = $_SESSION['rol'];
$nombre_usuario = $_SESSION['username'];
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Productos</title>
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <link rel="stylesheet" href="../CSS/prodcutos.css">
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary shadow-sm py-2">
        <div class="container-fluid">
            <button class="btn btn-link text-white d-lg-none" id="sidebar-toggle">
                <i class="bi bi-list fs-4"></i>
            </button>
            <a class="navbar-brand ms-2" href="#">
                <i class="bi bi-box-seam fs-4 me-2"></i>
                <span class="fw-bold">Gestión de Productos</span>
            </a>
            <div class="ms-auto d-flex align-items-center">
                <div class="dropdown">
                    <button class="btn btn-primary dropdown-toggle d-flex align-items-center" type="button" id="userDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                        <img src="../Img/admin.jpg" alt="Admin" width="32" height="32" class="rounded-circle me-2">
                        <span><?php echo htmlspecialchars($nombre_usuario); ?></span>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userDropdown">
                        <li><a class="dropdown-item" href="#">Mi perfil</a></li>
                        <li><a class="dropdown-item" href="#">Configuración</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="../logout.php">Cerrar sesión</a></li>
                    </ul>
                </div>
            </div>
        </div>
    </nav>

    <!-- Overlay -->
    <div class="overlay" id="sidebar-overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="d-flex flex-column p-4">
            <div class="text-center mb-4">
                <img src="../Img/admin.jpg" alt="Perfil" class="rounded-circle mb-2" width="80" height="80">
                <h6 class="text-white"><?php echo htmlspecialchars($nombre_usuario); ?></h6>
                <span class="badge bg-light text-primary"><?php echo ucfirst(htmlspecialchars($rol_usuario)); ?></span>
            </div>

            <ul class="nav flex-column mt-2">
                <li class="nav-item">
                    <a class="nav-link" href="../Admin/dashboard.php">
                        <i class="bi bi-house-door me-2"></i> Dashboard
                    </a>
                </li>
                
                <?php if ($_SESSION['rol'] === 'admin'): ?>
                <li class="nav-item">
                    <a class="nav-link" href="../Admin/usuarios.php">
                        <i class="bi bi-people me-2"></i> Usuarios
                    </a>
                </li>
                <?php endif; ?>
                
                <li class="nav-item">
                    <a class="nav-link" href="../Admin/carrusel.php">
                        <i class="bi bi-images me-2"></i> Carrusel
                    </a>
                </li>
                
                <li class="nav-item">
                    <a class="nav-link active" href="../Admin/productos.php">
                        <i class="bi bi-box-seam me-2"></i> Productos
                    </a>
                </li>
                
                <li class="nav-item">
                    <a class="nav-link" href="../index.php">
                        <i class="bi bi-display me-2"></i> Ver sitio
                    </a>
                </li>
                
                <li class="nav-item mt-5">
                    <a class="nav-link bg-danger text-white" href="../logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i> Cerrar sesión
                    </a>
                </li>
            </ul>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content content-wrapper">
        <div class="container-fluid">
            <header class="d-flex justify-content-between align-items-center py-3 mb-4 border-bottom">
                <h2 class="h2 fw-bold">Gestión de Productos</h2>
            </header>

            <!-- Mensajes de éxito o error -->
            <?php if(isset($success_message)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($success_message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            
            <?php if(isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <!-- Formulario para crear producto -->
            <div class="card shadow mb-4">
                <div class="card-header bg-primary bg-opacity-10 py-3">
                    <h5 class="card-title mb-0 fw-bold">Crear Nuevo Producto</h5>
                </div>
                <div class="card-body">
                    <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="crear_producto" value="1">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="nombre" class="form-label">Nombre del Producto</label>
                                <input type="text" name="nombre" id="nombre" class="form-control" required>
                            </div>
                            <div class="col-md-3">
                                <label for="precio" class="form-label">Precio</label>
                                <div class="input-group">
                                    <span class="input-group-text">Q</span>
                                    <input type="number" name="precio" id="precio" step="0.01" min="0" class="form-control" required>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <label for="stock" class="form-label">Stock</label>
                                <input type="number" name="stock" id="stock" min="0" class="form-control" required>
                            </div>
                            <div class="col-md-6">
                                <label for="imagen" class="form-label">Imagen</label>
                                <input type="file" name="imagen" id="imagen" accept="image/*" class="form-control">
                            </div>
                            <div class="col-md-6">
                                <div class="form-check mt-4" hidden>
                                    <input type="checkbox" name="destacado" id="destacado" class="form-check-input">
                                    <label for="destacado" class="form-check-label">Producto Destacado</label>
                                </div>
                            </div>
                            <div class="col-12">
                                <label for="descripcion" class="form-label">Descripción</label>
                                <textarea name="descripcion" id="descripcion" rows="4" class="form-control" required></textarea>
                            </div>
                            <div class="col-12 mt-3">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-plus-circle me-2"></i>Crear Producto
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Lista de productos -->
            <div class="card shadow">
                <div class="card-header bg-primary bg-opacity-10 py-3">
                    <h5 class="card-title mb-0 fw-bold">Productos Registrados</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover align-middle">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Imagen</th>
                                    <th>Nombre</th>
                                    <th>Precio</th>
                                    <th>Stock</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($producto = mysqli_fetch_assoc($result_productos)): ?>
                                <tr>
                                    <td><?php echo $producto['id']; ?></td>
                                    <td>
                                        <?php if (!empty($producto['imagen'])): ?>
                                            <img src="../img/productos/<?php echo htmlspecialchars($producto['imagen']); ?>" alt="<?php echo htmlspecialchars($producto['nombre']); ?>" class="product-image">
                                        <?php else: ?>
                                            <div class="no-image">Sin imagen</div>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($producto['nombre']); ?></td>
                                    <td>Q<?php echo number_format($producto['precio'], 2); ?></td>
                                    <td><?php echo $producto['stock']; ?></td>
                                    
                                    <td>
                                        <div class="action-buttons">
                                            <a href="editar_producto.php?id=<?php echo $producto['id']; ?>" class="btn btn-sm btn-primary">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="POST" onsubmit="return confirm('¿Estás seguro de que deseas eliminar este producto?');" class="d-inline">
                                                <input type="hidden" name="eliminar_producto" value="1">
                                                <input type="hidden" name="producto_id" value="<?php echo $producto['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                                
                                <?php if (mysqli_num_rows($result_productos) == 0): ?>
                                <tr>
                                    <td colspan="7" class="text-center py-4">No hay productos registrados</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap 5 JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom JS -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Sidebar toggle functionality
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');
            const overlay = document.getElementById('sidebar-overlay');
            
            if (sidebarToggle && sidebar && overlay) {
                sidebarToggle.addEventListener('click', function() {
                    sidebar.classList.toggle('show');
                    overlay.classList.toggle('show');
                });
                
                overlay.addEventListener('click', function() {
                    sidebar.classList.remove('show');
                    overlay.classList.remove('show');
                });
            }
            
            // Close sidebar when clicking on a nav link (mobile only)
            const navLinks = document.querySelectorAll('.sidebar .nav-link');
            navLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth < 992) {
                        sidebar.classList.remove('show');
                        overlay.classList.remove('show');
                    }
                });
            });
            
            // Handle window resize
            window.addEventListener('resize', function() {
                if (window.innerWidth >= 992) {
                    sidebar.classList.remove('show');
                    overlay.classList.remove('show');
                }
            });
        });
    </script>
</body>
</html>